/* job.c */
/* Job execution and handling for GNU Make.
Copyright (C) 1988-1991 Free Software Foundation, Inc.
This file is part of GNU Make.

GNU Make is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Make is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Make; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "make.h"
#include "commands.h"
#include "job.h"
#include "file.h"
#include "variable.h"
#include "var_func.h"
#include <errno.h>
#include <process.h>

extern int errno;

#if	defined(POSIX) || defined(__GNU_LIBRARY__)
#include <limits.h>
#include <unistd.h>
#define	GET_NGROUPS_MAX	sysconf (_SC_NGROUPS_MAX)
#else	/* Not POSIX.  */
#ifndef	USG
#include <sys/param.h>
#define	NGROUPS_MAX	NGROUPS
#endif	/* Not USG.  */
#endif	/* POSIX.  */

#if	defined(__GNU_LIBRARY__) || defined(POSIX)

#include <sys/types.h>

#else	/* Not GNU C library.  */

extern int dup2 ();
extern void _exit ();
extern int geteuid (), getegid ();
extern int setgid (), getgid ();
#endif	/* GNU C library.  */

#ifndef USG
extern int getdtablesize ();
#else
#include <sys/param.h>
#define getdtablesize() NOFILE
#endif

void init_siglist ();



/* Nonzero if the `good' standard input is in use.  */


/* Write an error message describing the exit status given in
   EXIT_CODE, EXIT_SIG, and COREDUMP, for the target TARGET_NAME.
   Append "(ignored)" if IGNORED is nonzero.  */

static void child_error (char *target_name, int exit_code, int ignored)
{
    error ("*** [%s] Error %d%s", target_name, exit_code
		, ignored ? " (ignored)" : "");
}


/* Make signals blocked in FLAG is nonzero, unblocked if FLAG is zero.
   Push this setting on the signals_blocked_p_stack, so it can be
   popped off by pop_signals_blocked_p.  */

void push_signals_blocked_p (int flag)
{
}

/* Pop the signals_blocked_p setting from the stack
   and block or unblock signals as appropriate.  */

void pop_signals_blocked_p(void)
{
}


/* Free the storage allocated for CHILD.  */


/* Start a job to run the commands specified in CHILD.
   CHILD is updated to reflect the commands and ID of the child process.  */

void start_job (struct child *child)
{
  	register char *p;
  	char noprint = 0, recursive;
  	argPtr argv;
   	int exit_code;
   	int child_failed;

  	if (child->command_ptr == 0 || *child->command_ptr == '\0')
   	{
      	/* There are no more lines in the expansion of this line.  */
      	if (child->command_line == child->file->cmds->ncommand_lines)
		{
	  		/* There are no more lines to be expanded.  */
	  		child->command_ptr = 0;
	  		child->file->command_state = cs_finished;
	  		child->file->update_status = 0;
	  		return;
		}
      	else
		{
	  		/* Get the next line to run, and set RECURSIVE
	     		if the unexpanded line contains $(MAKE).  */
	  		child->command_ptr = child->command_lines[child->command_line];
	  		recursive = child->file->cmds->lines_recurse[child->command_line];
	  		++child->command_line;
		}
   	}
  	else
    	/* Still executing the last line we started.  */
    	recursive = child->file->cmds->lines_recurse[child->command_line - 1];

  	/* Find the end of this line.  Backslash-newlines don't mean the end.  */

  	p = child->command_ptr;
  	child->noerror = 0;
  	while (*p != '\0')
   	{
      	if (*p == '@')
			noprint = 1;
      	else if (*p == '-')
			child->noerror = 1;
      	else if (*p == '+')
			recursive = 1;
      	else if (!isblank (*p))
			break;
      	++p;
   	}

  	/* If -q was given, just say that updating `failed'.  */
  	if (question_flag && !recursive)
    	goto error;

  	/* There may be some preceding whitespace left if there
     	was nothing but a backslash on the first line.  */
  	p = next_token (p);

  	/* Figure out an argument list from this command line.  */

  	{
    	char *end;
    	argv = construct_command_argv (p, &end, child->file);
    	if (end == NULL)
	      	child->command_ptr = NULL;
    	else
      	{
			*end++ = '\0';
			child->command_ptr = end;
      	}
  	}

  	/* Print out the command.  */

  	if (just_print_flag || (!noprint && !silent_flag))
    	puts (p);

  	if (!argv)
   	{
      	/* This line has no commands.  Go to the next.  */
      	start_job (child);
      	return;
   	}

  	/* If -n was given, recurse to get the next line in the sequence.  */

  	if (just_print_flag && !recursive)
   	{
		argPtr_free(argv);
		start_job(child);
		return;
   	}

  	/* Flush the output streams so they won't have things written twice.  */

  	fflush (stdout);
  	fflush (stderr);
  
  	child->deleted = 0;

  	/* Set up the environment for the child.  */
  	if (child->environment == 0)
    	child->environment = target_environment (child->file);

	exit_code = child_execute_job (argv, child->environment);
   	child_failed = exit_code != 0;

	if (child_failed && !child->noerror && !ignore_errors_flag)
    {
	    /* The commands failed.  Write an error message,
		delete non-precious targets, and abort.  */
	    child_error (child->file->name, exit_code, 0);
	    child->file->update_status = 1;
    }
	else
    {
	    if (child_failed)
		{
			/* The commands failed, but we don't care.  */
			child_error (child->file->name, exit_code, 1);
			child_failed = 0;
		}
    }
	start_job(child);	/* Play it again sam... */

	if (child->file->update_status != 0)
	{
		/* We failed to start the commands.  */
		delete_child_targets (child);
	}

	/* Set the state flag to say the commands have finished.  */
	notice_finished_file (child->file);

	/* If the job failed, and the -k flag was not given, die.  */
	if (child_failed && !keep_going_flag)
	    die (1);

  	/* Free the storage used by the child's argument list.  */

	argPtr_free(argv);
  	return;

error:;
  	child->file->update_status = 1;
}


/* Create a `struct child' for FILE and start its commands running.  */

void new_job (struct file *file)
{
  	register struct commands *cmds = file->cmds;
  	struct child c;
  	char **lines;
  	register int i;

  	/* Chop the commands up into lines if they aren't already.  */
  	chop_commands (cmds);

  	/* Expand the command lines and store the results in LINES.  */
  	lines = (char **) xmalloc (cmds->ncommand_lines * sizeof (char *));
  	for (i = 0; i < cmds->ncommand_lines; ++i)
    	lines[i] = allocated_variable_expand_for_file (cmds->command_lines[i],
						   	file);

  	/* Start the command sequence, record it in a new
     	`struct child', and add that to the chain.  */

	memset(&c, 0, sizeof(c));
  	c.file = file;
  	c.command_lines = lines;
  	c.command_line = 0;
  	c.command_ptr = 0;
  	c.environment = 0;
  	start_job (&c);

  	if (c.command_lines != 0)
    {
      	register int i;
      	for (i = 0; i < c.file->cmds->ncommand_lines; ++i)
			xfree (c.command_lines[i]);
      	xfree ((char *) c.command_lines);
    }

  	if (c.environment != 0)
   	{
      	register char **ep = c.environment;
      	while (*ep != 0)
			xfree (*ep++);
      	xfree ((char *) c.environment);
   	}
}

/* Replace the current process with one executing the command in ARGV.
   STDIN_FD and STDOUT_FD are used as the process's stdin and stdout; ENVP is
   the environment of the new program.  This function does not return.  */

int child_execute_job (argPtr argv, char **envp)
{
	return exec_command(argv, envp);
}

/* Figure out the argument list necessary to run LINE as a command.
   Try to avoid using a shell.  This routine handles only ' quoting.
   Starting quotes may be escaped with a backslash.  If any of the
   characters in sh_chars[] is seen, or any of the builtin commands
   listed in sh_cmds[] is the first word of a line, the shell is used.

   If RESTP is not NULL, *RESTP is set to point to the first newline in LINE.
   If *RESTP is NULL, newlines will be ignored.

   SHELL is the shell to use, or nil to use the default shell.
   IFS is the value of $IFS, or nil (meaning the default).  */
/* The original GNU make of course could pass everything off to the shell,
   but we don't have that luxury because we have to assume that neither dos
   nor os/2 under emx have /bin/sh.  Pity, but true..  */

static argPtr construct_command_argv_internal (char *line, char **restp)
{
	argPtr argv;
  	char *p, *ap, *end;
  	int instring;
	char arg[512];

	argv = argPtr_new(strlen (line) + 1);

  	ap = arg;
  	end = arg + sizeof(arg);

  	if (restp != NULL)
    	*restp = NULL;

  	instring = 0;
  	for (p = line; *p != '\0';)
   	{
      	if (ap > end)
			abort ();

      	if (instring)
		{
	  		/* Inside a string, just copy any char except a closing quote.  */
	  		if (*p == '\"')
	    		instring = 0;
    		*ap++ = *p++;
			continue;
		}

		switch (*p)
	  	{
		case '#':
			while (*p != '\0' && *p != '\n')
				p++;
			if (*p == '\n')
				*p = ' ';
			break;

		case '\'':
			*ap++ = *p++;
			while (*p != '\0' && *p != '\n' && *p != '\'')
			{
				if (*p == '\\' && p[1] == '\'')
					*ap++ = *p++;
				*ap++ = *p++;
			}
			break;

// At this point we should have already taken care of '`' !!!!!
		case '`':
			*ap++ = *p++;
			while (*p != '\0' && *p != '\n' && *p != '\'')
			{
				if (*p == '\\' && p[1] == '\'')
					*ap++ = *p++;
				*ap++ = *p++;
			}
			break;

	  	case '\\':
	    	/* Backslash-newline combinations are eaten.  */
	    	if (p[1] != '\0' && p[1] != '\n')
			{
	      		/* Copy and skip the following char.  */
	      		*ap++ = *p++;
				*ap++ = *p++;
			}
			else if (p[1] == '\n')
				p++;
	    	break;

	  	case '\"':
	    	instring = 1;
			*ap++ = *p++;
	    	break;

	  	case '\n':
	    	if (restp != NULL)
	      	{
				/* End of the command line.  */
				*restp = p;
				goto end_of_line;
	      	}
	    	else
			{
		      	/* Newlines are not special.  */
	    		*ap++ = '\0';
				argPtr_add(argv, ap = arg);
				p = next_token(p);
			}
	    	break;

		case '(':
		case ')':
		case ';':
	    	*ap++ = '\0';
			argPtr_add(argv, ap = arg);
			*ap++ = *p++;
			*ap++ = '\0';
			argPtr_add(argv, ap = arg);
	    	p = next_token (p);
			break;

		case '<':
		case '>':
		case '&':
		case '|':
			*ap++ = '\0';
			argPtr_add(argv, ap = arg);
			while (*p == '<' || *p == '>' || *p == '&' || *p == '|')
				*ap++ = *p++;
			*ap++ = '\0';
			argPtr_add(argv, ap = arg);
			p = next_token(p);
			break;

	  	case ' ':
	  	case '\t':
	    	*ap++ = '\0';
			argPtr_add(argv, ap = arg);
	    	p = next_token (p);
	    	/* Next iteration should examine the first nonwhite char.  */
	    	break;

	  	default:
	    	*ap++ = *p++;
	    	break;
	  	}
   	}

end_of_line:
  	/* Terminate the last argument and the argument list.  */

  	*ap = '\0';
	argPtr_add(argv, ap = arg);

  	if (!argv->ptrs[0])
	{
		argPtr_free(argv);
    	return NULL;
	}
   	return argv;
}


/* Figure out the argument list necessary to run LINE as a command.
   Try to avoid using a shell.  This routine handles only ' quoting.
   Starting quotes may be escaped with a backslash.  If any of the
   characters in sh_chars[] is seen, or any of the builtin commands
   listed in sh_cmds[] is the first word of a line, the shell is used.

   If RESTP is not NULL, *RESTP is set to point to the first newline in LINE.
   If *RESTP is NULL, newlines will be ignored.

   FILE is the target whose commands these are.  It is used for
   variable expansion for $(SHELL) and $(IFS).  */

argPtr construct_command_argv (char *line, char **restp, struct file *file)
{
  return construct_command_argv_internal (line, restp);
}

void init_siglist (void)
{
	return;
}
